/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * This program uses an implemenation of MD5 by Ulrich Drepper that
 * was also released under the terms of the GPL
 *
 * Version 0.6
 */


#define USERSFILE "/etc/security/motp.conf"

#define STATFILE "/var/cache/motp"

#define MAX_TRIES 5

#define MAX_DIFF 360

/*
  secret   = init string created by #**#
  passcode = string user types in
  PIN      = user's pin
  otp      = calculated by server, must match passcode
*/


#define LEN_SECRET    32
#define LEN_PASSCODE   6
#define LEN_PIN        4
#define LEN_OTP        LEN_PASSCODE



/* partly stolen from pam_pwdfile.c */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>

#include <security/pam_appl.h>

#define PAM_SM_AUTH
#define PAM_SM_ACCOUNT
#define PAM_SM_PASSWORD
#include <security/pam_modules.h>

#ifdef SOLARIS
#define PAM_EXTERN
#endif

#ifndef SOLARIS
#include "md5.h"
#endif

#define TO_STR(str) TO_STR_2(str)
#define TO_STR_2(str) #str

#ifdef _TRACE_
#define TRACE fprintf
#else
#define TRACE //
#endif

#define BUFSIZE 256


#define _PAM_LOG if (!(flags & PAM_SILENT)) _pam_log
/* logging function ripped from pam_listfile.c */
void _pam_log(int err, const char *format, ...) {
  va_list args; 
  va_start(args, format);
  openlog("pam_mobile_otp", LOG_CONS|LOG_PID, LOG_AUTH);
  vsyslog(err, format, args);
  // vfprintf(stderr, format, args ); fputc('\n',stderr);
  va_end(args);
  closelog();
}




#define NOT_IMPLEMENTED(fct) \
     _pam_log (LOG_ERR, "error: " #fct " is not implemented by pam_mobile_otp")

PAM_EXTERN int pam_sm_open_session(pam_handle_t *pamh, int flags,
				   int argc, const char **argv) {
  NOT_IMPLEMENTED (session management);
  return PAM_SERVICE_ERR;
}

PAM_EXTERN int pam_sm_close_session(pam_handle_t *pamh, int flags,
				    int argc, const char **argv) {
  NOT_IMPLEMENTED (session management);
  return PAM_SERVICE_ERR;
}



PAM_EXTERN int pam_sm_authenticate(pam_handle_t *pamh, int flags,
                                   int argc, const char **argv) {    
  int retval, i;
  const char *usersfile = USERSFILE;
  const char *statfile = STATFILE;
  int maxdiff = MAX_DIFF;
  int maxtries = MAX_TRIES;

  char secret[LEN_SECRET+1], passcode[LEN_PASSCODE+1], PIN[LEN_PIN+1];
  short offset;
  const char *user;
  char buffer [BUFSIZE+1];
  long unsigned int now;

  short debug = 0;
  short no_warn = 0;
  short use_first_pass = 0;
  short try_first_pass = 0;
  short not_set_pass = 0;

  char * otp ( long unsigned int now, char *secret, char *PIN ) {
    unsigned char md5sum[16];
    int i;
    snprintf ( buffer, BUFSIZE, "%0ld%s%s", now, secret, PIN );
#ifdef SOLARIS
    md5_calc ( md5sum, buffer, strlen(buffer) );
#else
    md5_buffer ( buffer, strlen(buffer), md5sum );
#endif
    for (i=0; i<LEN_OTP/2 && i<=sizeof(md5sum); i++ )
      sprintf( buffer+2*i, "%02x", md5sum[i] );
    buffer[LEN_OTP]=0;
    TRACE(stderr,"now=%ld, secret=%s, PIN=%s, otp=%s\n",now,secret,PIN,buffer);
    return buffer;
  }

  
  /* read options */
  for (i=0; i<argc; i++) {  
    if (strncmp(argv[i],"users=",strlen("users="))==0) {
      usersfile = strchr( argv[i], '=');
      usersfile++;
    }
    else if (strncmp(argv[i],"cache=",strlen("cache="))==0) {
      statfile = strchr( argv[i], '=');
      statfile++;
    }
    else if (strncmp(argv[i],"maxtries=",strlen("maxtries="))==0) {
      char *p = strchr( argv[i],'='); p++;
      sscanf( p, "%d", &maxtries );
    }
    else if (strncmp(argv[i],"maxdiff=",strlen("maxdiff="))==0) {
      char *p = strchr( argv[i],'='); p++;
      sscanf( p, "%d", &maxdiff );
    }
    else if (strcmp(argv[i],"debug")==0)
      debug=1;
    else if (strcmp(argv[i],"no_warn")==0)
      no_warn=1;
    else if (strcmp(argv[i],"use_first_pass")==0)
      use_first_pass=1;
    else if (strcmp(argv[i],"try_first_pass")==0)
      try_first_pass=1;
    else if (strcmp(argv[i],"not_set_pass")==0)
      not_set_pass=1;
    else
      _PAM_LOG(LOG_ERR,"option not implemented: %s",argv[i]);
  }


  /* read user name */
  if ((retval=pam_get_user(pamh,&user,NULL)) != PAM_SUCCESS) {
    _PAM_LOG(LOG_ERR, "error: get username");
    return retval;
  }
  TRACE(stderr,"USER: %s\n",user);
  

  /* get passcode */
 get_passcode:
  if (use_first_pass || try_first_pass) {
    char *p;
    retval = pam_get_item(pamh,PAM_AUTHTOK,(void *)&p);
    if (retval != PAM_SUCCESS) return PAM_AUTHINFO_UNAVAIL;
    if (p) { strncpy(passcode, p ,LEN_PASSCODE); passcode[LEN_PASSCODE]=0; }
    TRACE(stderr,"old passcode: %s\n",passcode);
    if (debug) _PAM_LOG(LOG_DEBUG, "using old passcode (%s)",
			use_first_pass?"use_first_pass":"try_first_pass");
  }
  else {
    struct pam_message msg, *pmsg[1];
    struct pam_response *resp;
    struct pam_conv *conv;
    msg.msg = "passcode: ";
    msg.msg_style = PAM_PROMPT_ECHO_OFF;
    pmsg[0] = &msg;
    
    retval = pam_get_item (pamh, PAM_CONV, (const void **) &conv);
    if (retval != PAM_SUCCESS) return retval;
    retval = conv->conv (1, (const struct pam_message **)pmsg, &resp, conv->appdata_ptr);
    if (retval != PAM_SUCCESS) return retval;
    if (!resp) return PAM_CONV_ERR;
    strncpy(passcode, resp->resp ,LEN_PASSCODE); passcode[LEN_PASSCODE]=0;
    if (!not_set_pass) pam_set_item (pamh, PAM_AUTHTOK, resp->resp);
    if (resp) free (resp);
  }
  TRACE(stderr,"user: %s, passcode: %s\n", user, passcode);
  
  
  /* read secret from users file */
  { 
    FILE *fp;
    char name[BUFSIZE+1]; name[0]=0;
    if ( (fp=fopen(usersfile,"r")) == NULL ) {
      _PAM_LOG(LOG_ALERT, "error: cannot open usersfile %s",usersfile);
      return PAM_AUTHINFO_UNAVAIL; 
    }
    while (fgets(buffer,BUFSIZE,fp)) {
      if ( (buffer[0]=='#') || (buffer[0]==0) ) continue;
      buffer[BUFSIZE]=0;
      TRACE(stderr,"buffer=%s\n",buffer);
      retval = sscanf(buffer, "%" TO_STR(BUFSIZE) "s %" TO_STR(LEN_SECRET) "s %" TO_STR(LEN_PIN) "s %hd ",
		      &name, &secret, &PIN, &offset );
      if (retval < 3) continue;
      if (retval == 3) offset=0;
      TRACE(stderr,"trying: user: %s, secret: %s, pin: %s, offset: %d -- user: %s\n", 
	    name, secret, PIN, offset, user);
      if ( strcmp(user,name)==0 ) break; 
    }
    if ( fclose(fp) ) if (!no_warn) 
      _PAM_LOG(LOG_INFO, "warning: cannot close usersfile %s",usersfile);
    if ( strcmp(user,name) ) {
      _PAM_LOG(LOG_INFO, "user %s not found in file %s", user,usersfile);
      return PAM_USER_UNKNOWN;
    }
    TRACE(stderr,"user: %s, secret: %s, pin: %s\n", name, secret, PIN);
    if (debug) _PAM_LOG(LOG_DEBUG, "user: %s", name);
  }
  
  
  /* passcode ok? */
  retval=PAM_AUTH_ERR;

  now = time(NULL);      
  now += 3600*offset;
  now /= 10;
  now -= maxdiff/10/2;
  i = now + maxdiff/10;
  for ( ; now <= i ; now++ )
    if (strcmp(passcode,otp(now,secret,PIN))==0) {
	if (debug) _PAM_LOG(LOG_DEBUG, "passcode accepted");
	retval=PAM_SUCCESS;
	break;
    }

  if ( (retval!=PAM_SUCCESS) && (try_first_pass) ) {
    TRACE(stderr,"trying without \"try_first_pass\"\n");
    try_first_pass = 0;
    goto get_passcode;
  }
  if (retval!=PAM_SUCCESS) _PAM_LOG(LOG_NOTICE, "passcode not accepted");
  TRACE (stderr, "passcode %saccepted: %s\n", (retval==PAM_SUCCESS?"":"not "), passcode);
  

  /* read and write user's status file */
  {
    FILE *fp;
    int tries = 0;
    long unsigned int last_time = 0;
    
    snprintf(buffer,BUFSIZE,"%s/%s",statfile,user);
    buffer[BUFSIZE]=0;
    statfile=buffer;
    if ( (fp=fopen(statfile,"r")) == NULL ) {
      if (debug) _PAM_LOG(LOG_DEBUG, "user %s has no statfile",user);
      tries = last_time = 0;
    }
    else {
      if ( fscanf(fp,"%d:%lu:%*s",&tries,&last_time) != 2 ) {
	_PAM_LOG(LOG_INFO, "cannot read data from statfile %s",statfile);
	fclose(fp);
	return PAM_AUTHINFO_UNAVAIL;
      }
      if (debug) _PAM_LOG(LOG_DEBUG, "user has tries: %d, last time: %lu", tries, last_time);
      if ( fclose(fp) ) if (!no_warn)
	_PAM_LOG(LOG_INFO, "warning: cannot close file (r) %s",statfile);
    }

    // check status
    {
      int codeok = (retval == PAM_SUCCESS);
      int replay = (!(now>last_time));
      int locked = (tries > maxtries);

      if ( locked )
	retval=PAM_MAXTRIES;
      if ( replay ) 
        retval=PAM_AUTH_ERR;

      if ( codeok && !replay && !locked ) // everything ok: reset counter
	tries = 0;
      if ( !codeok )  // wrong passcode: increase counter
        tries++;
      if ( codeok && !replay )  // no replay
	last_time = now;

      if (debug) {
	if (replay) _PAM_LOG(LOG_DEBUG, "replay detected (time: %lu)", now);
        if (locked) _PAM_LOG(LOG_DEBUG, "user locked (more than %d unsuccessful attempts)",maxtries);
        _PAM_LOG(LOG_DEBUG, "new tries: %d, time: %lu", tries, last_time);
      }
    }

    umask(077);
    if ( (fp=fopen(statfile,"w")) == NULL ) {
      _PAM_LOG(LOG_ERR, "cannot write statfile %s",statfile);
      return PAM_AUTHINFO_UNAVAIL;
    }
    if ( fprintf(fp,"%d:%lu:%s\n",tries,last_time,"") < 0 ) {
      _PAM_LOG(LOG_INFO, "cannot write new statfile %s",statfile);
      retval = PAM_AUTHINFO_UNAVAIL;
    }
    TRACE(stderr,"written new stats - tries: %d, last time: %lu\n",tries,last_time);
    if ( fclose(fp) ) if (!no_warn)
      _PAM_LOG(LOG_INFO, "warning: cannot close file (w) %s",statfile);
  }
  

  /* return pam code */
  return retval;
}


PAM_EXTERN int pam_sm_setcred(pam_handle_t *pamh, int flags,
                              int argc, const char **argv)
{
  return PAM_SUCCESS;
}


PAM_EXTERN int pam_sm_chauthtok(pam_handle_t *pamh, int flags, 
				int argc, const char **argv) {
  
  /*
    Function is not thread save at the moment.
    Moreover, changing usersfile while user changes pin is problematic.
    User must not type in old pin :-( 
  */
  
  int retval, i;
  const char *usersfile = USERSFILE;
  char PIN[LEN_PIN+1];  
  const char *user;
  char buffer[BUFSIZE+1], name[BUFSIZE+1];
  int pos = 0;
  FILE *fp;
  
  short debug = 0;
  short no_warn = 0;
  //short use_first_pass = 0;
  //short try_first_pass = 0;
  //short not_set_pass = 0;

  /* read options */
  for (i=0; i<argc; i++) {
    if (strncmp(argv[i],"users=",strlen("users="))==0) {
      usersfile = strchr( argv[i], '=');
      usersfile++;
    }
    else if (strcmp(argv[i],"debug")==0)
      debug=1;
    else if (strcmp(argv[i],"no_warn")==0)
      no_warn=1;
    else
      _PAM_LOG(LOG_ERR,"option not implemented: %s",argv[i]);
  }
  
  
  /* read user name */
  if ((retval=pam_get_user(pamh,&user,NULL)) != PAM_SUCCESS) {
    _PAM_LOG(LOG_ERR, "error: get username");
    return retval;
  }


  if (debug) _PAM_LOG(LOG_DEBUG, "user %s tries to change PIN - %s",
		      user, (flags & PAM_PRELIM_CHECK) ? "1st run" : "2nd run" );


  if (flags & PAM_PRELIM_CHECK) {
    /* read secret from users file */
    {
      char *p;
      name[0]=0;
      if ( (fp=fopen(usersfile,"r")) == NULL ) {
	_PAM_LOG(LOG_ALERT, "error: cannot open usersfile %s",usersfile);
	return PAM_TRY_AGAIN;
      }
      while (fgets(buffer,BUFSIZE,fp)) {
	pos += strlen(buffer);
	if ( (buffer[0]=='#') || (buffer[0]==0) ) continue;
	buffer[BUFSIZE]=0;
	if ( sscanf(buffer, "%" TO_STR(BUFSIZE) "s %*s %" TO_STR(LEN_PIN) "s",  &name, &PIN ) !=2 ) continue;
	if ( strcmp(user,name)==0 ) break;
      }
      if ( fclose(fp) ) if (!no_warn)
	_PAM_LOG(LOG_INFO, "warning: cannot close usersfile %s",usersfile);
      if ( strcmp(user,name) ) {
	_PAM_LOG(LOG_INFO, "user %s not found in file %s", user,usersfile);
	pos=0;
	return PAM_USER_UNKNOWN;
      }
      pos -= strlen(buffer);
      p=buffer;
      while ( p && (! isspace(*p)) ) p++;  // skip name
      while ( p &&    isspace(*p)  ) p++;  // skip gap
      while ( p && (! isspace(*p)) ) p++;  // skip secret
      while ( p &&    isspace(*p)  ) p++;  // skip gap
      pos += (p-buffer);
      retval = pam_set_data (pamh, "motp-position", (void *)pos, NULL);
      if (retval != PAM_SUCCESS) return PAM_TRY_AGAIN;
      TRACE(stderr,"position: %d\n",pos);
    }
    return PAM_SUCCESS;
  }


  /* get new PIN */
  {
    struct pam_message msg, *pmsg[1];
    struct pam_response *resp;
    struct pam_conv *conv;
    int tries = 0;

    void message ( pam_handle_t *pamh, char *str ) {
      struct pam_response *mresp;
      msg.msg_style = PAM_ERROR_MSG;
      msg.msg = str;
      retval = pam_get_item (pamh, PAM_CONV, (const void **) &conv);
      if (retval != PAM_SUCCESS) return;
      retval = conv->conv (1, (const struct pam_message **)pmsg,
			   &mresp, conv->appdata_ptr);
      if (mresp) free (mresp);
    }

    pmsg[0] = &msg;

  new_pin:
    if (++tries > 3) return PAM_AUTHTOK_ERR;

    msg.msg_style = PAM_PROMPT_ECHO_OFF;
    msg.msg = "new PIN: ";
    retval = pam_get_item (pamh, PAM_CONV, (const void **) &conv);
    if (retval != PAM_SUCCESS) return retval;
    retval = conv->conv (1, (const struct pam_message **)pmsg, &resp, conv->appdata_ptr);
    if (retval != PAM_SUCCESS) return retval;
    if (!resp) return PAM_CONV_ERR;
    strncpy(PIN, resp->resp ,LEN_PIN); PIN[LEN_PIN]=0;
    TRACE(stderr,"new pin: %s\n",PIN);

    if ( strlen(resp->resp) > 4 )
      message(pamh, "PIN will be truncated to 4 digits");
    free (resp);

    if (strlen(PIN)<4) {
      message(pamh, "PIN must be 4 digits long -- please retry\n");
      goto new_pin;
    }

    for (i=0;i<strlen(PIN);i++)
      if (! ( isdigit(PIN[i]) || (PIN[i]=='#') || (PIN[i]=='*') ) ) {
	message(pamh, "PIN must only contain digits, '#' or '*' -- please retry\n");
	goto new_pin;
      }
    
    if (strcmp(PIN,"#**#")==0) {
      message(pamh, "PIN not allowed -- please retry\n");
      goto new_pin;
    }
    TRACE(stderr,"pin ok\n");

    msg.msg_style = PAM_PROMPT_ECHO_OFF;
    msg.msg = "new PIN again: ";
    pmsg[0] = &msg;
    retval = pam_get_item (pamh, PAM_CONV, (const void **) &conv);
    if (retval != PAM_SUCCESS) return retval;
    retval = conv->conv (1, (const struct pam_message **)pmsg, &resp, conv->appdata_ptr);
    if (retval != PAM_SUCCESS) return retval;
    if (!resp) return PAM_CONV_ERR;

    if ( strlen(resp->resp) > 4 )
      message(pamh, "PIN will be truncated to 4 digits");
    
    if ( strncmp(PIN, resp->resp, strlen(PIN)) ) {
      free (resp);
      message(pamh, "PINs are different -- please retry");
      goto new_pin;
    }
    free (resp);
  }
  

  /* open usersfile for writing and change PIN */
  retval = pam_get_data (pamh, "motp-position", (const void **) &pos);
  if (retval != PAM_SUCCESS) return retval;
  if ( (fp=fopen(usersfile,"r+")) == NULL ) {
    _PAM_LOG(LOG_ALERT, "error: cannot open usersfile %s",usersfile);
    return PAM_PERM_DENIED;
  }
  TRACE(stderr,"position: %d\n",pos);
  for (;pos--;) if (fgetc(fp)==EOF) return PAM_PERM_DENIED;
  fflush(fp);
  if ( fputs(PIN,fp) == EOF ) return PAM_PERM_DENIED;
  if ( fclose(fp) ) if (!no_warn)
    _PAM_LOG(LOG_INFO, "warning: cannot close usersfile %s",usersfile);
  
  if (debug) _PAM_LOG(LOG_DEBUG, "PIN changed" );
  return PAM_SUCCESS;
}


PAM_EXTERN int pam_sm_acct_mgmt(pam_handle_t *pamh, int flags,
                                int argc, const char **argv) {    
  int retval, i;
  const char *statfile = STATFILE;

  const char *user;
  char buffer [BUFSIZE+1];

  short debug = 0;
  short no_warn = 0;


  /* read options */
  for (i=0; i<argc; i++) {  
    if (strncmp(argv[i],"cache=",strlen("cache="))==0) {
      statfile = strchr( argv[i], '=');
      statfile++;
    }
    else if (strcmp(argv[i],"debug")==0)
      debug=1;
    else if (strcmp(argv[i],"no_warn")==0)
      no_warn=1;
    else
      _PAM_LOG(LOG_ERR,"option not implemented: %s",argv[i]);
  }


  /* read user name */
  if ((retval=pam_get_user(pamh,&user,NULL)) != PAM_SUCCESS) {
    _PAM_LOG(LOG_ERR, "error: get username");
    return retval;
  }
  TRACE(stderr,"USER: %s\n",user);
  

  /* read and write user's status file */
  {
    FILE *fp;
    int tries = 0;
    long unsigned int last_time = 0;
    
    snprintf(buffer,BUFSIZE,"%s/%s",statfile,user);
    buffer[BUFSIZE]=0;
    statfile=buffer;

    /* read user's status file */
    if ( (fp=fopen(statfile,"r")) == NULL ) {
      if (debug) _PAM_LOG(LOG_DEBUG, "user %s has no statfile",user);
      tries = last_time = 0;
    }
    else {
      if ( fscanf(fp,"%d:%lu:%*s",&tries,&last_time) != 2 ) {
	_PAM_LOG(LOG_INFO, "cannot read data from statfile %s",statfile);
	fclose(fp);
	return PAM_SUCCESS;
      }
      if ( fclose(fp) ) if (!no_warn)
	_PAM_LOG(LOG_INFO, "warning: cannot close file (r) %s",statfile);

      /* reset retry counter */
      tries = 0;
      
      /* write user's status file */
      umask(077);
      if ( (fp=fopen(statfile,"w")) == NULL ) {
	_PAM_LOG(LOG_ERR, "cannot write statfile %s",statfile);
	return PAM_SERVICE_ERR;
      }
      if ( fprintf(fp,"%d:%lu:%s\n",tries,last_time,"") < 0 ) {
	_PAM_LOG(LOG_INFO, "cannot write statfile %s",statfile);
	retval = PAM_SERVICE_ERR;
      }
      TRACE(stderr,"Has reset tries and written new stats - tries: %d, last time: %lu\n",tries,last_time);
      if ( fclose(fp) ) if (!no_warn)
	_PAM_LOG(LOG_INFO, "warning: cannot close file (w) %s",statfile);
    }
  }
  

  /* return pam code */
  return retval;
}


#ifdef PAM_STATIC
struct pam_module _pam_mobile_otp_modstruct = {
  "pam_mobile_otp",
  pam_sm_authenticate,
  pam_sm_setcred,
  pam_sm_acct_mgmt,
  NULL,
  NULL,
  pam_sm_chauthtok,
};
#endif

