<?php
######################################################################
# Weather Magnet
# ============================================
# Copyright (c) 2003 by John Horne seventhseal@ibdeeming.com
# http://ibdeeming.com
# This program is free software. You can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License.
######################################################################
if (!defined('CPG_NUKE')) { exit; }
$module_name = basename(dirname(__FILE__));
get_lang($module_name);

require_once("modules/$module_name/wh_ini.php");
require_once("modules/$module_name/wh_utils.php");
require_once("modules/$module_name/wh_block.php");

/******************************************************************************/
/* function fnCacheCleanup()                                                  */
/* This function cleans out the cache directory of old files                  */
/******************************************************************************/

function fnCacheCleanup( $local_file_path ) {
  global $db, $prefix;
  $wh = new whini();

  if ($handle = opendir($local_file_path))
  {
    while (false !== ($file = readdir($handle)))
	{ 
      if ($file != "." && $file != "..")
	  { 
	    $now = time();
	    $age = filectime( $local_file_path.$file );
		$cachetime = $wh->getcachetimeout(); 
		if ( $now > ( $age + $cachetime ) )  // see if it's too old
		{
		  unlink( $local_file_path.$file );  // it's too old, delete it!!!
		}
      } 
    }
    closedir($handle); 
  }
}

/******************************************************************************/
/* function processMETAR( $local_file )                                       */
/* This function processes METAR data generated by NOAA.                      */
/* The file passed in is the local filename on your host server.              */
/******************************************************************************/

function processMETARFile( $local_file, $mini_block )
{
  $lines = "";

  if (file_exists( $local_file ) )
  {
    $handle = fopen( $local_file, "r" );
    while (!feof($handle) )
    {
      $lines .= fgets( $handle, 1024 );
    }
    fclose( $handle );

    if ($lines != "")
    {
      $return_string = processMETARLines( $lines, $mini_block );
    }
    else
    {
      $return_string = _WEATHER_NOT_AVAILABLE;
    }
  }
  else
  {
    $return_string = _WEATHER_NOT_AVAILABLE;
  }

  return( $return_string );
}

/******************************************************************************/
/* processWindDirection( $direction )                                         */
/*function processes the degrees direction and returns alpha direction of wind*/
/******************************************************************************/

function processWindDirection( $direction )
{
  if ( $direction > 10 && $direction <= 35 )  // NNE
  {
    $dirOne = "NNE"; 
  }
  else if ( $direction > 35 && $direction <= 55 ) // NE
  {
    $dirOne = "NE";
  }
  else if ($direction > 55 && $direction <= 80) // ENE
  {
    $dirOne = "ENE";
  }
  else if ($direction > 80 && $direction <= 100 ) // E
  {
    $dirOne = "E";
  }
  else if ($direction > 100 && $direction <= 125 ) // ESE
  {
    $dirOne = "ESE";
  }
  else if ( $direction > 125 && $direction <= 145 ) // SE
  {
    $dirOne = "SE";
  }
  else if ($direction > 145 && $direction <= 170 ) // SSE
  {
    $dirOne = "SSE";
  }
  else if ($direction > 170 && $direction <= 190 ) // S
  {
    $dirOne = "S";
  }
  else if ($direction > 190 && $direction <= 215 ) // SSW
  {
    $dirOne = "SSW";
  }
  else if ($direction > 215 && $direction <= 235 ) // SW
  {
    $dirOne = "SW";
  }
  else if ($direction > 235 && $direction <= 260 ) // WSW
  {
    $dirOne = "WSW";
  }
  else if ( $direction > 260 && $direction <= 280 ) // W
  {
    $dirOne = "W";
  }
  else if ($direction > 280 && $direction <= 305 ) // WNW
  {
    $dirOne = "WNW";
  }
  else if ( $direction > 305 && $direction <= 325 ) // NW
  {
    $dirOne = "NW";
  }
  else if ($direction > 325 && $direction <= 350 )  // NNW
  {
    $dirOne = "NNW";
  }
  else  // N
  {
    $dirOne = "N";
  }
  
  return( $dirOne );
}

/******************************************************************************/
/* processMETARLines( $lines, $mini_block )                                   */
/* processes the standard METAR data as defined by NOAA.  $lines is the data, */
/* $mini_block is flag that overrides display and only shows small block on side*/
/******************************************************************************/

function processMETARLines( $lines, $mini_block )
{
  global $db, $prefix;
  $whblock = new whblock();
  
  $alignment = ( $whblock->getblockwidth() == "0" ? "right" : "left" );
 
  if ( $whblock->gettimepos() == "T" )
  {
    $return_string = weatherDateTime( $lines );
  }
  	
  
  $tempWindString = "";

/****************************** Begin Wind speed ******************************/
// get the wind speed and direction
  if ( eregi( "00000KT", $lines ) ) // look for calm first
  {
    $tempWindString = "<b>Wind:</b></td><td align=\"$alignment\">Calm<br />";
	$wndMPH = 0;
	$wndGMPH = 0;
	$wndkmh = 0;
  }
  else if ( eregi( "VRB([0-9]*)KT", $lines, $buffer ) ) // look for variable wind with VRB
  {
    $wndKT = $buffer[1];
      // convert to mph
    $wndMPH = $wndKT * 1.143;
	$wndkmh = $wndMPH * 1.6;
	switch( $whblock->getmeasuretype() )
	{
	  case "0":
	  default:
        $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">Variable at %.02f kt (%.02f kmh / %.02f mph)", $wndKT, $wndkmh, $wndMPH );
	    break;
		  
	  case "1": // standard  
        $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">Variable at %.02f mph", $wndMPH );
		break;
		  
	  case "2":  // metric
        $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">Variable at %.02f kmh", $wndkmh );
		break;
	}
	  	  
    $tempWindString = $wndString;
  }
  else if ( eregi( "([0-9]{5})KT ([0-9]{3})V([0-9]{3})", $lines, $buffer ) ) // look for variable direction with directionals
  {
      // get direction first
    $directionOne = intval( substr( $buffer[2], 0, 3 ) );
    $directionTwo = intval( substr( $buffer[3], 0, 3 ) );
    $wndKT = intval( substr( $buffer[1], 3, 2 ) );
      // convert speed to mph
    $wndMPH = $wndKT * 1.143;
    $wndkmh = $wndMPH * 1.6;
	  
	$dirOne = processWindDirection( $directionOne );
	$dirTwo = processWindDirection( $directionTwo );
	  
	switch( $whblock->getmeasuretype() )
	{
	  case "0":
	  default:
	    $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">Variable from the %s to the %s at %.02f kt (%.2f kmh / %.02f mph)", $dirOne, $dirTwo, $wndKT, $wndkmh, $wndMPH );
	    break;
		  
	  case "1": // standard  
	    $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">Variable from the %s to the %s at %.02f mph", $dirOne, $dirTwo, $wndMPH );
	    break;
		  
	  case "2": // metric
	    $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">Variable from the %s to the %s at %.2f kmh", $dirOne, $dirTwo, $wndkmh );
		break;  
	}

    $tempWindString = $wndString;
  }
  else if (eregi( "([0-9]{5})KT G([0-9]{2})KT ", $lines, $buffer ) )   // another form of gusting wind
  {
      // get direction first
    $direction = intval( substr( $buffer[1], 0, 3 ) );
    $wndKT = intval( substr( $buffer[1], 3, 2 ) );
    $wndGKT = intval( substr( $buffer[2], 0, 2 ) );
      // convert speed to mph
    $wndMPH = $wndKT * 1.143;
	$wndkmh = $wndMPH * 1.6;
    $wndGMPH = $wndGKT * 1.143;
	$wndGkmh = $wndGMPH * 1.6;

    $dir = processWindDirection( $direction );
    
	switch( $whblock->getmeasuretype() )
	{
	  case "0":
	  default:
	    $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">From the %s at %.02f kt (%.2f kmh / %.02f mph) gusting to %.02f kt (%.2f kmh / %.02f mph)", $dir, $wndKT, $wndkmh, $wndMPH, $wndGKT, $wndGkmh, $wndGMPH );
	    break;
		  
	  case "1": // standard
	    $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">From the %s at %.02f mph gusting to %.02f mph", $dir, $wndMPH, $wndGMPH );
	    break;
		  
	  case "2":  // metric
		$wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">From the %s at %.2f kmh gusting to %.2f kmh", $dir, $wndkmh, $wndGkmh );
		break;    
	}

    $tempWindString = $wndString;
  }
  else if ( eregi( "([0-9]{5})KT ", $lines, $buffer ) )  // just look for standard wind measurement
  {
    // get direction first
    $direction = intval( substr( $buffer[1], 0, 3 ) );
    $wndKT = intval( substr( $buffer[1], 3, 2 ) );
    // convert speed to mph
    $wndMPH = $wndKT * 1.143;
    $wndkmh = $wndMPH * 1.6;
	 
    $dir = processWindDirection( $direction );

    switch( $whblock->getmeasuretype() )
	{
	  case "0":
	  default:
        $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">From the %s at %.02f kt (%.2f kmh / %.02f mph)", $dir, $wndKT, $wndkmh, $wndMPH );
	    break;
		  
	  case "1": // standard
        $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">From the %s at %.02f mph", $dir, $wndMPH );
	    break;
		  
	  case "2":  // metric
        $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">From the %s at %.2f kmh", $dir, $wndkmh );
	    break;
    }    
    
    $tempWindString = $wndString;
  }
  else if ( eregi( "([0-9]{5}G[0-9]{2})KT ", $lines, $buffer ) )  // just look for gusting wind measurement
  {
    // get direction first
    $direction = intval( substr( $buffer[1], 0, 3 ) );
    $wndKT = intval( substr( $buffer[1], 3, 2 ) );
    $wndGKT = intval( substr( $buffer[1], 6, 2 ) );
    // convert speed to mph
    $wndMPH = $wndKT * 1.143;
	$wndkmh = $wndMPH * 1.6;
    $wndGMPH = $wndGKT * 1.143;
	$wndGkmh = $wndGMPH * 1.6;
 
    $dir = processWindDirection( $direction );
	  
	switch( $whblock->getmeasuretype() )
	{
	  case "0":
	  default:
        $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">From the %s at %.02f kt (%.02f kmh / %.02f mph) gusting to %.02f kt (%.02f kmh / %.02f mph)", $dir, $wndKT, $wndkmh, $wndMPH,  $wndGKT, $wndGkmh,$wndGMPH );
	    break;
		  
	  case "1": // standard
        $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">From the %s at %.02f mph gusting to %.02f mph", $dir, $wndMPH, $wndGMPH );
		break;
		  
  	  case "2":  // metric
        $wndString = sprintf( "<b>Wind:</b></td><td align=\"$alignment\">From the %s at %.02f kmh gusting to %.02f kmh", $dir, $wndkmh, $wndGkmh );
  	    break;
    }

    $tempWindString = $wndString;
  }
  else
  {
    $tempWindString = "</td><td>";
  }
/*************************** end wind speed ***********************************/

  $return_string = weatherCurCond( $lines, $mini_block, isset($_GET['wndMPH']), isset($_GET['wndGMPH']) );

  if ($mini_block == false)
  {
    // get current conditions

    $return_string .= "<table><tr><td align=\"left\">\n";

    $return_string .= $tempWindString;

    $return_string .= "</td></tr>\n<tr><td align=\"left\">";

  // get barometric pressure
    if ( eregi( " A([0-9]{4})", $lines, $buffer ) ) 
    {
      $bpIn = substr( $buffer[1], 0, 4 );
      $bpIn /= 100.0;
    // figure out hPa
      $bphPa = ( $bpIn * 3386.389 ) / 100.0;
      
	  // determine pressure tendency
	  $pres_tend = "steady";
	  if ( eregi( " 5([0-9])([0-9]{3})", $lines, $buffer ) )
	  {
	    // we have a pressure tendency...
		switch( $buffer[1] )
		{
		  case "0":  // it's up and down
		    $pres_tend = "up and down";
		    break;
			
	      case "1":  // very slow increase
		    $pres_tend = "very slow increase";
		    break;
			
		  case "2":  // steady increase
		    $pres_tend = "steady increase";
		    break;
			
		  case "3":  // increasing fast
		    $pre_tend = "increasing fast";
		    break;
			
		  case "4": // steady
		    $pres_tend = "steady";
		    break;
			
	      case "5":  // up and down at measures - but steady
		    $pres_tend = "up and down over time, but steady overall";
		    break;
			
		  case "6":  // very slow decrease
		    $pres_tend = "very slow decrease";
		    break;
			
		  case "7":  // steady decrease
		    $pres_tend = "steady decrease";
		    break;
			
		  case "8":  // decreasing fast
		    $pres_tend = "decreasing fast";
		    break;							
		}
	  }
	  
	  switch( $whblock->getmeasuretype() )
	  {
	    case "0":
		default:
          $bpString = sprintf( "<b>Pres.:</b></td><td align=\"$alignment\"> %.02f hPa (%.02f in.)", $bphPa, $bpIn );
		  break;
		  
		case "1": // standard
		  $bpString = sprintf( "<b>Pres.:</b></td><td align=\"$alignment\"> %.02f in.", $bpIn );
		  break;
		  
		case "2":  // metric
		  $bpString = sprintf( "<b>Pres.:</b></td><td align=\"$alignment\"> %.02f hPa", $bphPa );
		  break;
	  }
	  
      $return_string .= $bpString."<br />".$pres_tend;
    }
    else if (eregi( " Q([0-9]{4})", $lines, $buffer ) ) 
    {
      $bphPa = substr( $buffer[1], 0, 4 );
    // figure out in.
      $bpIn = ( $bphPa * 100.0 ) / 3386.389;
	  
	  // determine pressure tendency
	  $pres_tend = "steady";
	  if ( eregi( " 5([0-9])([0-9]{3})", $lines, $buffer ) )
	  {
	    // we have a pressure tendency...
		switch( $buffer[1] )
		{
		  case "0":  // it's up and down
		    $pres_tend = "up and down";
		    break;
			
	      case "1":  // very slow increase
		    $pres_tend = "very slow increase";
		    break;
			
		  case "2":  // steady increase
		    $pres_tend = "steady increase";
		    break;
			
		  case "3":  // increasing fast
		    $pre_tend = "increasing fast";
		    break;
			
		  case "4": // steady
		    $pres_tend = "steady";
		    break;
			
	      case "5":  // up and down at measures - but steady
		    $pres_tend = "up and down over time, but steady overall";
		    break;
			
		  case "6":  // very slow decrease
		    $pres_tend = "very slow decrease";
		    break;
			
		  case "7":  // steady decrease
		    $pres_tend = "steady decrease";
		    break;
			
		  case "8":  // decreasing fast
		    $pres_tend = "decreasing fast";
		    break;							
		}
	  }
	  
      switch( $whblock->getmeasuretype() )
	  {
	    case "0":
		default:
          $bpString = sprintf( "<b>Pres.:</b></td><td align=\"$alignment\">%.2f hPa (%.02f in.)", $bphPa, $bpIn );
		  break;
		  
		case "1": // standard
		  $bpString = sprintf( "<b>Pres.:</b></td><td align=\"$alignment\">%.02f in.", $bpIn );
		  break;
		  
		case "2": // metric
          $bpString = sprintf( "<b>Pres.:</b></td><td align=\"$alignment\">%.02f hPa", $bphPa ); 		
		  break;
	  }
	  
      $return_string .= $bpString."<br />".$pres_tend;
    }
    else
    {
      $return_string .= "</td><td>";
    }


    $return_string .= "</td></tr>\n<tr><td align=\"left\">";

  // get visibility info
    if ( eregi( "(M [0-9]/[0-9]*)SM ", $lines, $buffer ) ) 
    {
      // found the M - so a number will follow, and the string should be less than x miles
      $visSM = substr( $buffer[1], 1 );
	  $visKM = $visSM * 1.6;
	  
	  switch( $whblock->getmeasuretype() )
	  {
	    case "0":
		default:
          $visString = sprintf( "<b>Visibility:</b></td><td align=\"$alignment\">Less than %s Km / %s miles", $visKM, $visSM );
		  break;
		  
		case "1": // standard
          $visString = sprintf( "<b>Visibility:</b></td><td align=\"$alignment\">Less than %s miles", $visSM );
		  break;
		  
		case "2": // metric
		  $visString = sprintf( "<b>Visibility:</b></td><td align=\"$alignment\">Less than %s Km", $visKM );
		  break;
	  } 	  
      $return_string .= $visString;
    }
    else if ( eregi( "([0-9] [0-9]/[0-9])SM ", $lines, $buffer ) ) // just take the string and format out...
    {
      $visSM = substr( $buffer[1], 0 );
	  $visKM = $visSM * 1.6;
	  
	  switch( $whblock->getmeasuretype() )
	  {
	    case "0":
		default:
		  $visString = sprintf( "<b>Visibility:</b></td><td align=\"$alignment\">%s Km / %s miles", $visKM, $visSM );
		  break;
		  
		case "1":  
          $visString = sprintf( "<b>Visibility:</b></td><td align=\"$alignment\">%s miles", $visSM );
		  break;
		  
		case "2":  
          $visString = sprintf( "<b>Visibility:</b></td><td align=\"$alignment\">%s Km", $visKM );
		  break;
	  }	  
      $return_string .= $visString;
    }
    else if ( eregi( "([0-9]*)SM ", $lines, $buffer ) ) // just take the string and format out...
    {
      $visSM = substr( $buffer[1], 0 );
	  $visKM = $visSM * 1.6;
	  
	  switch( $whblock->getmeasuretype() )
	  {
	    case "0":
		default:
          $visString = sprintf( "<b>Visibility:</b></td><td align=\"$alignment\">%s Km / %s miles", $visKM, $visSM );
		  break;
		  
		case "1":  
          $visString = sprintf( "<b>Visibility:</b></td><td align=\"$alignment\">%s miles", $visSM );
		  break;
		  
		case "2":  
          $visString = sprintf( "<b>Visibility:</b></td><td align=\"$alignment\">%s Km", $visKM );
		  break;
	  }	  
      $return_string .= $visString;
    }
    else
    {
      $return_string .= "</td><td>";
    }

    $return_string .= "</td></tr></table>";
  }
  
  if ( $whblock->gettimepos() == "B" )
  {
    $return_string .= weatherDateTime( $lines );
  }
  	
  return $return_string;
}

function weatherDateTime( $lines )
{
  global $db, $prefix;
  
  $wh = new whblock();
  
  $return_string = "";
  // get the date and time string
  if ( eregi( "([0-9]{2})([0-9]{2})([0-9]{2})Z", $lines, $buffer ) )
  {
    $timeofday = gettimeofday();
    $day = $buffer[1];
	$zhourtime = $buffer[2];
	$zmintime = $buffer[3];
    $dst = ( $timeofday['dsttime'] == 0 ? 0 : 1 );
	$datenow = getdate( time() );
    $ztime = gmmktime( $zhourtime, $zmintime, 0, $datenow['mon'], $day, $datenow['year'], $dst );
	$timezonestr = ( $wh->gettimezone() === true ? " %Z" : "" );
	$formatted = "";
	$yearDone = false;
	
	switch( $wh->getdatedisplay() )  // Get date format
	{
	  case "0": // MMM DD,
	  default:
	   	$formatted = strftime( "%b %d, ", $ztime );
        break;
		
      case "1":  // DD MMM,
	    $formatted = strftime( "%d %b, ", $ztime );
	    break;
		
	  case "2": // , MMM DD
	    $yearDone = true;
	    switch( $wh->getyeardisplay() )
		{
	      case "0": // YYYY
	      default:
	        $formatted = strftime( "%Y ", $ztime );
	        break;
		
	      case "1": // YY
	        $formatted = strftime( "%y ", $ztime );
	        break;	
		}
	    $formatted .= strftime( ", %b %d", $ztime );
	    break;			
	}
	
	if ( $yearDone == false )
	{
	  switch( $wh->getyeardisplay() )  // get year format
	  {
	    case "0": // YYYY
	    default:
	      $formatted .= strftime( "%Y ", $ztime );
	      break;
		
	    case "1": // YY
	      $formatted .= strftime( "%y ", $ztime );
	      break;	
	  }
	}
	
	switch( $wh->gettimedisplay() )  // get time format
	{
	  case "0":  // 12 hour
	    $formatted .= strftime( "%H:%M".$timezonestr, $ztime );
		break;
		
      case "1": // 24 hour
	  default:
	    $formatted .= strftime( "%I:%M %p".$timezonestr, $ztime );
		break;
   }			
	
    $return_string = "<table><tr><td>"._REPORTED_ON."<br />".$formatted."</td></tr></table>";
  }

  return $return_string;
}

function weatherCurCond( $lines, $mini_block, $wndMPH, $wndGMPH = "" ) {
  global $db, $prefix, $module_name;

  $wh = new whini();
  $whblock = new whblock();
  
  switch( $whblock->getfontsize() )
  {
    case "S":
	  $fontsize = "8";
	  break;
	  
	case "M":
	default:
	  $fontsize = "10";
	  break;
	  
	case "L":
	  $fontsize = "12";
	  break;
  }
  
  $alignment = ( $whblock->getblockwidth() == "0" ? "right" : "left" );
  $wimages_loc = "modules/WeatherHarvest/images/set".$wh->getimageset()."/";
  
  $lines = strchr( $lines, ' ' );

  // figure out the time of day
  $setDawntime = $whblock->getdawntime();
  $setDusktime = $whblock->getdusktime();
  
  $timeofday = localtime( time(), 1 );
  if ( $timeofday["tm_hour"] >= ($setDawntime/100.0) && $timeofday["tm_hour"] <= ($setDusktime/100.0) )
  {
    $isDayTime = true;
  }
  else
  {
    $isDayTime = false;
  }
  
  $w_condition = preg_split('/\s+/', $lines );

  $w_image = "";

  while (list(, $tok ) = each( $w_condition ) )
  {
    if ( preg_match('/OVC|BKN|SCT|FEW|CLR|SKC|CAVOK/', $tok ) )
    {

    // look for the cloud condition
      $cloudCond = "";
      if (eregi( "OVC", $tok ) ) 
      {
        $cloudCond = "Overcast";
        if ( $w_image == "" )
        {
          $w_image = "26.gif";
        }
      }
      else if (eregi("BKN", $tok ) ) 
      {
        $cloudCond = "Mostly Cloudy";
        if ( $w_image == "" ) 
        {
          if ( $isDayTime == true )
          {
            $w_image = "28.gif";
          }
          else
          {
            $w_image = "27.gif";
          }  
        }
      }
      else if (eregi( "SCT", $tok ) ) 
      {
        $cloudCond = "Partly Cloudy";

        if ( $w_image == "" ) 
        {
          if ( $isDayTime == true )
          {
            $w_image = "30.gif";
          }
          else
          {
            $w_image = "29.gif";
          }  
        }
      }
      else if (eregi( "FEW", $tok ) ) 
      {
        $cloudCond = "Mostly Clear";

        if ( $w_image == "" ) 
        {
          if ( $isDayTime == true )
          {
            $w_image = "34.gif";
          }
          else
          { 
            $w_image = "33.gif";
          }  
        }
      }
      else if (eregi( "CLR", $tok ) || eregi( "SKC", $tok ) )
      {
        $cloudCond = "Clear";

        if ( $w_image == "" ) 
        {
          if ( $isDayTime == true )
          {
            $w_image = "32.gif";
          }
          else
          {
            $w_image = "31.gif";
          }  
        }
      }
      else if (eregi( "CAVOK", $tok ) ) 
      {
        $cloudCond = "Fair";


        if ( $w_image == "" ) 
        {
          if ( $isDayTime == true )
          {
            $w_image = "32.gif";
          }
          else
          {
            $w_image = "31.gif";
          }  
        }
      }
    }
    else if (preg_match('/^(-|\+|VC)?(NSW|TS|SH|FZ|BL|DR|MI|BC|PR|RA|DZ|SN|SG|GR|GS|PE|IC|UP)+$/', $tok)) 
    {

        // look for the weather condition
        // look for intensity
      $intensity = "";

      if ( preg_match( "/^(\+)/", $tok ) )  // + = heavy
      {
        $intensity = "Heavy";
      }
      else if ( preg_match( "/^(-)/", $tok ) ) // - = light
      {
        $intensity = "Light";
      }
      else if ( eregi( "/^(VC)/", $tok ) )  // VC = vicinity
      {
        $intensity = "In the vicinity";
      }
      else
      {
        $intensity = "Moderate";
      }

        // get the descriptor
      $descriptor = "";
      if ( eregi( "MI", $tok ) )
      {
        $descriptor = "shallow";
      }
      else if (eregi( "PR", $tok ) ) 
      {
        $descriptor = "partial";
      }
      else if (eregi( "BC", $tok ) ) 
      {
        $descriptor = "patches";
      }
      else if (eregi( "DR", $tok ) ) 
      {
        $descriptor = "drifting";
      }
      else if (eregi( "BL", $tok ) ) 
      {
        $descriptor = "blowing";
      }
      else if (eregi( "SH", $tok ) ) 
      {
        $descriptor = "showers";
      }
      else if (eregi( "TS", $tok ) )
      {
        $descriptor = "thunderstorms";
      }
      else if (eregi( "FZ", $tok ) )
      {
        $descriptor = "freezing";
      }

        // get the precipitation
      $precipitation = "";
      if (eregi( "DZ|RA", $tok ) ) 
      {
        $precipitation .= " drizzle";
		if ( $descriptor == "freezing" )
		{
		  $w_image = "10.gif";
		}
		else
		{
          $w_image = "12.gif";
		}  
      }

      if (eregi( "SN", $tok ) ) 
      {
        $precipitation .= " snow";
		if ( $descriptor == "blowing" )
		{
		  $w_image = "43.gif";
		}
		else
		{
          $w_image = "14.gif";
		}  
      }

      if (eregi( "SG", $tok ) ) 
      {
        $precipitation = "snow grains";
        $w_image = "8.gif";
      }
      else if (eregi( "IC", $tok ) ) 
      {
        $precipitation = "ice crystals";
        $w_image = "10.gif";
      }
      else if (eregi( "IP", $tok ) ) 
      {
        $precipitation = "sleet";
        $w_image = "11.gif";
      }
      else if (eregi( "PL", $tok ) ) 
      {
        $precipitation = "ice pellets";
        $w_image = "11.gif";
      }
      else if (eregi( "GR", $tok ) ) 
      {
        $precipitation = "hail";
        $w_image = "11.gif";
      }
      else if (eregi( "GS", $tok ) ) 
      {
        $precipitation = "small ice or hail";
        $w_image = "6.gif";
      }
      else if (eregi( "UP", $tok ) ) 
      {
        $precipitation = "various precipitation";
      }
    }
    else if (preg_match('/(BR|FG|FU|VA|DU|SA|HZ|PY|\/)+$/', $tok) )
    {
      // get obscuration
      $obscuration = "";
      if (eregi( "BR", $tok ) ) 
      {
        $obscuration = "mist";
      }
      else if (eregi( "FG", $tok ) ) 
      {
        $obscuration = "fog";
        $w_image = "20.gif";
      }
      else if (eregi( "FU", $tok ) ) 
      {
        $obscuration = "smoke";
        $w_image = "22.gif";
      }
      else if (eregi( "VA", $tok ) ) 
      {
        $obscuration = "volcanic ash";
        $w_image = "22.gif";
      }
      else if (eregi( "DU", $tok ) ) 
      {
        $obscuration = "widespread dust";
        $w_image = "19.gif";
      }
      else if (eregi( "SA", $tok ) ) 
      {
        $obscuration = "sand";
        $w_image = "19.gif";
      }
      else if (eregi( "HZ", $tok ) ) 
      {
        $obscuration = "haze";
        $w_image = "21.gif";
      }
      else if (eregi( "PY", $tok ) ) 
      {
        $obscuration = "spray";
        $w_image = "24.gif";
      }
    }
    else if (preg_match('/(PO|SQ|FC|SS|DS|\/)+$/', $tok)  )
    {
        // get other
      $other = "";
      if (eregi( "PO", $lines ) ) 
      {
        $other = "well developed Dust/Sand Storm";
        $w_image = "19.gif";
      }
      else if (eregi( "SQ", $lines ) ) 
      {
        $other = "squalls";
        $w_image = "24.gif";
      }
      else if (eregi( "FC", $lines ) ) 
      {
        $other = "tornado, funnel cloud or waterspout";
      }
      else if (eregi( "SS", $lines ) ) 
      {
        $other = "sandstorm";
        $w_image = "19.gif";
      }
      else if (eregi( "DS", $lines ) ) 
      {
        $other = "duststorm";
        $w_image = "19.gif";
      }
    }
  } // end while

  if ( $w_image == "" ) 
  {
    $w_image = "-.gif";
  }

  $return_string = "";
  
  // get the temperature and dew point
  if ( eregi( " T([0-9]{8})", $lines, $buffer ) )
  {
    $negtempC = substr( $buffer[1], 0, 1 );
    $tempC = substr( $buffer[1], 1, 3 );
    $negdewC = substr( $buffer[1], 4, 1 );
    $dewC = substr( $buffer[1], 5, 3 );

    if ( $negtempC != 0 )
    {
      $tempC *= (-1.0);
    }

    $tempC /= 10.0;

    if ( $negdewC != 0 )
    {
      $dewC *= (-1.0);
    }

    $dewC /= 10.0;
  }
  else if ( eregi( " ([0-9]{2})/([0-9]{2}) ", $lines, $buffer ) )
  { 
    $tempC = $buffer[1];
    $dewC = $buffer[2];
  }
  else if (eregi( " ([0-9]{2})/M([0-9]{2})", $lines, $buffer ) ) 
  {
    $tempC = $buffer[1];
    $dewC = 0.0 - intval($buffer[2]);
  }
  else if (eregi( "M([0-9]{2})/M([0-9]{2})", $lines, $buffer ) )  // both are negative
  {
    $tempC = 0.0 - intval($buffer[1]);
    $dewC = 0.0 - intval($buffer[2]);
  }

  // C to F conversion
  $tempF = ( $tempC * 1.8 ) + 32.0;
  $dewF = ( $dewC * 1.8 ) + 32.0;

  // figure out relative humidity
  $satvapPres = 6.11 * pow( 10,((7.5*$tempC/(237.7+$tempC))));//saturation vapor pressure
  $actvapPres = 6.11 * pow( 10,((7.5*$dewC/(237.7+$dewC))));//actual vapor pressure
  $relHumidity = ($actvapPres / $satvapPres )* 100;//relative humidity
  
  // figure out Heat index (temp must be above 80 F
  $showHI = false;
  if ( $tempF >= 80.0 )
  {
    $showHI = true;  // display Heat Index
    $heatindexF = (-42.379) + ( 2.04901523 * $tempF ) + ( 10.1433127 * $relHumidity ) - ( 0.22475541 * $tempF * $relHumidity ) - ( .00683783 * pow( $tempF, 2 ) ) - ( .05481717 * pow( $relHumidity, 2 ) ) + ( .00122874 * pow( $tempF, 2 ) * $relHumidity ) + ( .00085282 * $tempF * pow( $relHumidity, 2 ) ) - ( .00000199 * pow( $tempF, 2 ) * pow( $relHumidity, 2 ) );
    $heatindexC = ( $heatindexF - 32.0 ) / 1.8;
    // figure out the danger level text
    if ( $heatindexF >= 80.0 && $heatindexF < 90.0 )
    {
      $HIDangerText = "<img src=\"modules/WeatherHarvest/images/c_hot.gif\" alt=\""._HI_CAUTION."\" />&nbsp;";
    }
    else if ( $heatindexF >= 90.0 && $heatindexF < 105.0 ) 
    {
      $HIDangerText = "<img src=\"modules/WeatherHarvest/images/ec_hot.gif\" alt=\""._HI_EX_CAUTION."\" />&nbsp;";
    }
    else if ( $heatindexF >= 105.0 && $heatindexF < 129.0 )
    {
      $HIDangerText = "<img src=\"modules/WeatherHarvest/images/d_hot.gif\" alt=\""._HI_DANGER."\" />&nbsp;";
    }
    else if ( $heatindexF >=129 )
    {
      $HIDangerText = "<img src=\"modules/WeatherHarvest/images/ed_hot.gif\" alt=\""._HI_EX_DANGER."\" />&nbsp;";
    }
  }

  // figure out Wind Chill temp must be below 40 F
  $showWC = false;
  $showGWC = false;
  if ( $tempF <= 40.0 && $wndMPH > 0 ) 
  {
    $showWC = true;
    $windchillF = 35.74 + ( 0.6215 * $tempF ) - ( 35.75 * pow( $wndMPH, 0.16 ) ) + ( 0.4275 * $tempF * pow( $wndMPH, 0.16 ) );
    $windchillC = ( $windchillF - 32.0 ) / 1.8;

    // figure out danger level
    if ( $windchillF <= 40.0 && $windchillF > 20.0 )
    {
      $WCDangerText = "<img src=\"modules/WeatherHarvest/images/c_cold.gif\" alt=\""._HI_CAUTION."\" />&nbsp;";
    }
    else if ( $windchillF <= 20.0 && $windchillF > 0.0 ) 
    {
      $WCDangerText = "<img src=\"modules/WeatherHarvest/images/ec_cold.gif\" alt=\""._HI_EX_CAUTION."\" />&nbsp;";
    }
    else if ( $windchillF <= 0.0 && $windchillF > (-18.0) )
    {
      $WCDangerText = "<img src=\"modules/WeatherHarvest/images/d_cold.gif\" alt=\""._HI_DANGER."\" />&nbsp;";
    }
    else if ( $windchillF <= (-18.0) )
    {
      $WCDangerText = "<img src=\"modules/WeatherHarvest/images/ed_cold.gif\" alt=\""._HI_EX_DANGER."\" />&nbsp;";
    }
	
	if ( $wndGMPH != "" && $wndGMPH > 0 )
	{
	  $showGWC = true;
	  $windchillGF = 35.74 + ( 0.6215 * $tempF ) - ( 35.75 * pow( $wndGMPH, 0.16 ) ) + ( 0.4275 * $tempF * pow( $wndGMPH, 0.16 ) );
      $windchillGC = ( $windchillGF - 32.0 ) / 1.8;

    // figure out danger level
      if ( $windchillGF <= 40.0 && $windchillGF > 20.0 )
      {
        $WCGDangerText = "<img src=\"modules/WeatherHarvest/images/c_cold.gif\" alt=\""._HI_CAUTION."\" />&nbsp;";
      }
      else if ( $windchillGF <= 20.0 && $windchillGF > 0.0 ) 
      {
        $WCGDangerText = "<img src=\"modules/WeatherHarvest/images/ec_cold.gif\" alt=\""._HI_EX_CAUTION."\" />&nbsp;";
      }
      else if ( $windchillGF <= 0.0 && $windchillGF > (-18.0) )
      {
        $WCGDangerText = "<img src=\"modules/WeatherHarvest/images/d_cold.gif\" alt=\""._HI_DANGER."\" />&nbsp;";
      }
      else if ( $windchillGF <= (-18.0) )
      {
        $WCGDangerText = "<img src=\"modules/WeatherHarvest/images/ed_cold.gif\" alt=\""._HI_EX_DANGER."\" />&nbsp;";
      }
	}
  }
  
  switch( $whblock->getmeasuretype() )
  {
    case "0":
	default:
      $tempString = sprintf( "%.02f&deg;C (%.02f&deg;F)<br />", $tempC, $tempF );
      
      if ($showHI == true)
      {
        $HIString = sprintf( "<tr><td align=\"left\">$HIDangerText<b>Heat Index:</b></td><td align=\"$alignment\">%.02f&deg;C (%.02f&deg;F)</td></tr>", $heatindexC, $heatindexF );
      }
      else
      {
        $HIString = "";
      }

      if ( $showWC == true )
      {
        $WCString = sprintf( "<tr><td align=\"left\">$WCDangerText<b>Wind Chill:</b></td><td align=\"$alignment\">%.02f&deg;C (%.02f&deg;F)</td></tr>", $windchillC, $windchillF );
		if ( $showGWC == true )
		{
		  $WCString .= sprintf( "<tr><td align=\"left\">$WCGDangerText<b>Gust Chill:</b></td><td align=\"$alignment\">%.02f&deg;C (%.02f&deg;F)</td></tr>", $windchillGC, $windchillGF );
		}
      }
      else
      {
        $WCString = "";
      }

      $dewString = sprintf( "<tr><td align=\"left\"><b>Dew Point:</b></td><td align=\"$alignment\">%.02f&deg;C (%.02f&deg;F)</td></tr>", $dewC, $dewF );
      break;
	  
	case "1": // standard only
      $tempString = sprintf( "%.02f&deg;F<br />", $tempF );
      
      if ($showHI == true)
      {
        $HIString = sprintf( "<tr><td align=\"left\">$HIDangerText<b>Heat Index:</b></td><td align=\"$alignment\">%.02f&deg;F</td></tr>", $heatindexF );
      }
      else
      {
        $HIString = "";
      }

      if ( $showWC == true )
      {
        $WCString = sprintf( "<tr><td align=\"left\">$WCDangerText<b>Wind Chill:</b></td><td align=\"$alignment\">%.02f&deg;F</td></tr>", $windchillF );
		if ( $showGWC == true )
		{
		  $WCString .= sprintf( "<tr><td align=\"left\">$WCGDangerText<b>Gust Chill:</b></td><td align=\"$alignment\">%.02f&deg;F</td></tr>", $windchillGF );
		}
      }
      else
      {
        $WCString = "";
      }

      $dewString = sprintf( "<tr><td align=\"left\"><b>Dew Point:</b></td><td align=\"$alignment\">%.02f&deg;F</td></tr>", $dewF );	
      break;  
	  
	case "2": // metric only
      $tempString = sprintf( "%.02f&deg;C<br />", $tempC );
      
      if ($showHI == true)
      {
        $HIString = sprintf( "<tr><td align=\"left\">$HIDangerText<b>Heat Index:</b></td><td align=\"$alignment\">%.02f&deg;C</td></tr>", $heatindexC );
      }
      else
      {
        $HIString = "";
      }

      if ( $showWC == true )
      {
        $WCString = sprintf( "<tr><td align=\"left\">$WCDangerText<b>Wind Chill:</b></td><td align=\"$alignment\">%.02f&deg;C</td></tr>", $windchillC );
		if ( $showGWC == true )
		{
		  $WCString .= sprintf( "<tr><td align=\"left\">$WCGDangerText<b>Gust Chill:</b></td><td align=\"$alignment\">%.02f&deg;C</td></tr>", $windchillGC );
		}
      }
      else
      {
        $WCString = "";
      }

      $dewString = sprintf( "<tr><td align=\"left\"><b>Dew Point:</b></td><td align=\"$alignment\">%.02f&deg;C</td></tr>", $dewC );
      break;  
  }
  	  
  $rhString = sprintf( "<tr><td align=\"left\"><b>Humidity:</b></td><td align=\"$alignment\">%.02f%s</td></tr>", $relHumidity, "%" );

  $iconalign = ( $whblock->getblockwidth() == 0 ? "center" : "left" );
  $return_string .= sprintf( "<table><tr><td align=\"$iconalign\"><img src='%s%s' alt=\"\" />", $wimages_loc, $w_image );
 
  // put reports together and return
  if ( $cloudCond != "" ) 
  {
    $return_string .= sprintf( "<br /><b>%s</b></td>", $cloudCond );
  }
  else
  {
    $return_string .= "</td>";
  }

  if ( $tempString != "" )
  {
    $return_string .= sprintf( "<td align=\"$iconalign\"><font size=\"".$fontsize."\">%s</font>", $tempString );
  }
  else
  {
    $return_string .= "<td>&nbsp;";
  }

  $return_string .= "</td></tr>\n";

  if ( isset($_GET['intensity']) != "" || isset($_GET['descriptor']) != "" || isset($_GET['precipitation']) != "" || isset($_GET['obscuration']) != "" || isset($_GET['other']) != "" ) {
    $return_string .= "<tr><td colspan='2'><b>Weather : </b>";

    if ( $intensity != "" || $descriptor != "" || $precipitation != "" ) 
    {
      $return_string .= sprintf( "%s %s %s", $intensity, $descriptor, $precipitation  );

      if ( $obscuration != "" ) 
      {
        $return_string .= sprintf( " / %s", $obscuration );
      }
    }
    else if ( $obscuration != "" ) 
    {
      $return_string .= sprintf( "%s", $obscuration );
    }
    else if ( $other != "" ) 
    {
      $return_string .= sprintf( "%s", $other );
    }
    $return_string .= "</td></tr>\n";
  }

  $return_string .= "</table>\n";

  if ($mini_block == false)
  {
    $return_string .= "<table>".$WCString.$HIString.$dewString.$rhString."</table>";
  }

  return $return_string;
}

?>